from mpf.system.mode import Mode
from mpf.system.tasks import DelayManager
from mpf.system.timing import Timing
import random

# door_independence                                                                                                                     Page 31
# Hinterlands Door Mode - "Fireworks Festival" Multiball                                                                                                             Page 26
#
# Brief Description
# Four major shots are lit for jackpots. Once scored, a travelling super jackpot can be obtained.
# 
# Scenario
# Uncle Sam's made it to Halloween Town and scared out of his wits... 
# so he comes up with a great idea to avoid being scared: Celebrate with Fireworks!
# 
# Details
# This is a 3-ball multiball which starts with four shots lit for jackpots: 
# Left Ramp, Right Ramp, Mayor and Soup. Once all jackpots are collected a travelling super
# jackpot appears. Once collected, the original four jackpot shots relight.
# 
# Scoring
# Jackpot                    2,000,000
# Super Jackpot              10,000,000
# 
# Lighting
# Jackpot shots flash their triangles red, then white, then blue, then white. It's VERY important not 
# to flash straight from red to blue or from blue to red as this may
# cause epileptic reactions in some people. The roaming super jackpot flashes both a white triangle and circle.
# 
# Difficulty Adjustments
# Very Easy        Super Jackpot Travels Slowest
# Easy             Super Jackpot Travels Slower
# Normal           Super Jackpot Travels Normal Speed
# Hard             Super Jackpot Travels Faster
# Very Hard        Super Jackpot Travels Fastest

class Door_Independence(Mode):

    def mode_init(self):
        self.log.info('door_independence mode_init')

    def mode_start(self, **kwargs):
        self.log.info('door_independence mode_start')
        self.delay = DelayManager()
        self.shot_value = 20000
        self.jackpot_value = 2000000
        self.super_jackpot_value = 10000000
        if self.player.door_independence_started == 0:
            #once per game only
            self.player.door_independence_started = 1
            self.player.door_independence_running = 0
            self.player.door_independence_score = 0
        self.player.door_independence_shot_index = 0
        self.player.door_independence_shot_direction = 1
        self.player.door_independence_shotlist = [
            {"led":"rgb_mayor_arrow", "mindex":0, "state":"off"}
            ,{"led":"rgb_lramp_arrow", "mindex":2, "state":"off"}
            ,{"led":"rgb_rramp_arrow", "mindex":7, "state":"off"}
            ,{"led":"rgb_soup_arrow", "mindex":9, "state":"off"}
            ]
        self.add_mode_event_handler("major_0_singlestep_unlit_hit", self.major_0)
        self.add_mode_event_handler("major_2_singlestep_unlit_hit", self.major_2)
        self.add_mode_event_handler("major_2a_singlestep_unlit_hit", self.major_2)
        self.add_mode_event_handler("major_7_singlestep_unlit_hit", self.major_7)
        self.add_mode_event_handler("major_7a_singlestep_unlit_hit", self.major_7)
        self.add_mode_event_handler("major_9_singlestep_unlit_hit", self.major_9)
        self.add_mode_event_handler('balldevice_trough_ball_enter', self.ball_drained)
        self.door_independence_start()


    def door_independence_start(self):
        self.log.info("door_independence_start")
        if (self.player.door_independence_running == 0):
            self.player.door_independence_running = 1
            self.player.door_independence_shots_made = 0
            self.set_shots()
            self.machine.events.post("holiday_independence_music_start")
            self.machine.events.post('disable_combos')
            bip = self.machine.game.balls_in_play 
            self.machine.game.add_balls_in_play(3-bip)
            self.machine.playfield.add_ball(balls=(3-bip), player_controlled=False)
            self.machine.events.post('add_a_ball_start')
            self.machine.events.post('enable_the_ball_save')


    def door_independence_stop(self):
        if self.player.door_independence_running == 1:
            self.machine.events.post('holiday_independence_music_stop')
            self.log.info("door_independence over")
            if self.player.door_independence_shots_made > 0:
                self.machine.events.post('holiday_mode_stopped', state="complete")
                self.player.door_independence_running = 2 #completed
            else:
                self.machine.events.post('holiday_mode_stopped', state="incomplete")
                self.player.door_independence_running = 0 #ready to start again
            self.reset_shots()
            self.machine.events.post('enable_combos')
            self.machine.events.post('add_a_ball_stop')


    def ball_drained(self, **kwargs):
        if self.player.door_independence_running == 1:
            self.log.info("door_independence - ball drained")
            self.log.info("door_independence - balls in play "+str(self.machine.game.balls_in_play))
            if self.machine.game.balls_in_play <= 2:
                if self.player.ball_save_active == 0:
                    #there is only 1 ball on playfield, end multiball
                    self.log.info("door_independence - Fireworks MB over")
                    self.door_independence_stop()
                else:
                    self.log.info( "door_independence multiball - ball drained - but ball save is running" )
            else:
                self.log.info( "door_independence multiball - ball drained - more than 2 BIP" )            
                

    def set_shots(self):
        self.player.door_independence_shots_qualified = 0
        for x in range(0, 4):
            self.player.door_independence_shotlist[x]["state"] = "redwhiteblue"
        self.set_shot_lights()
        self.log.info('door_independence - shots set') 


    def set_shot_lights(self):
        for x in range(0, 4):
            self.machine.light_controller.stop_script(key="door_independence_"+str(x))
        for x in range(0, 4):
            state = self.player.door_independence_shotlist[x]["state"]
            led = self.player.door_independence_shotlist[x]["led"]
            script_name = "sc_"+state
            scp = self.machine.light_controller.registered_light_scripts[script_name]
            self.machine.light_controller.run_script(leds=led, script=scp, priority=140, tocks_per_sec=60, key="door_independence_"+str(x), blend=True)

    def major_0(self, **kwargs):
        self.handle_shot(0)
    def major_2(self, **kwargs):
        self.handle_shot(1)
    def major_7(self, **kwargs):
        self.handle_shot(2)
    def major_9(self, **kwargs):
        self.handle_shot(3)

 
    def handle_shot(self, shot):
        self.log.info('door_independence - handle_shot')
        if self.player.door_independence_running == 1:
            self.score = 0
            state = self.player.door_independence_shotlist[shot]["state"]
            mindex = self.player.door_independence_shotlist[shot]["mindex"]
            if state != "off":
                if state == "redwhiteblue":
                    #qualifier
                    self.log.info('door_independence - jackpot')                                    
                    self.player.door_independence_shotlist[shot]["state"] = "off"
                    self.score = self.jackpot_value * self.player.multiplier_shot_value_list[mindex] 
                    self.player.door_independence_shots_qualified += 1
                    self.machine.events.post('holiday_independence_jackpot', value=self.score)                                            
                    self.player.door_independence_shots_made += 1
                    if self.player.door_independence_shots_qualified == 4:
                        self.log.info('door_independence - start the super jackpot!')
                        self.cycle_shots()
                elif state == "red_flash":
                    #super jackpot
                    self.log.info('door_independence - super jackpot')
                    self.player.door_independence_shotlist[shot]["state"] = "off"                    
                    self.delay.remove('door_independence_shot_cycler')
                    self.score = self.super_jackpot_value * self.player.multiplier_shot_value_list[mindex] 
                    self.machine.events.post('door_independence_sjackpot_hit', value=self.score)
                    self.set_shots()                    
                else:
                    # "off" - already collected
                    self.log.info('door_independence - points')                    
                    self.score = self.shot_value * self.player.multiplier_shot_value_list[mindex] 
                    self.machine.events.post('door_independence_points_hit', value=self.score)
                self.set_shot_lights()
                self.player.score += self.score
                self.player.door_independence_score += self.score

        

    def cycle_shots(self):
        self.log.info('door_independence - cycle_shots')
        if self.player.door_independence_running == 1:
            for x in range(0, 4):
                self.player.door_independence_shotlist[x]["state"] = "off"
            self.player.door_independence_shot_index += self.player.door_independence_shot_direction
            if self.player.door_independence_shot_index > 3:
                self.player.door_independence_shot_index = 2
                self.player.door_independence_shot_direction = -1
            if self.player.door_independence_shot_index < 0:
                self.player.door_independence_shot_index = 1
                self.player.door_independence_shot_direction = 1
            self.player.door_independence_shotlist[self.player.door_independence_shot_index]["state"] = "red_flash"
            delaytime = Timing.string_to_ms('500ms')
            self.delay.add(name="door_independence_shot_cycler", ms=delaytime, callback=self.cycle_shots)
            self.set_shot_lights()
            self.log.info('door_independence - cycle_shots 500ms delay added')
            self.machine.events.post('holiday_independence_cycling')            
            


    def reset_shots(self):
        self.delay.remove('door_independence_shot_cycler')
        for x in range(0, 4):
            self.player.door_independence_shotlist[x]["state"] = "off"
        self.set_shot_lights()


    def mode_stop(self, **kwargs):
        self.log.info('door_independence_mode_stop')
        self.door_independence_stop()
        self.reset_shots()

