from mpf.system.mode import Mode
from mpf.system.tasks import DelayManager
from mpf.system.timing import Timing

# Character Mode - "Kidnap the Sandi-Claws" Multiball
# 
# Brief Description
# Similar multiball to CFTBL. Three random major shots are lit, but only one has 
# Santa Claus in it. Once found, you have to get him back to Jack in the Graveyard 
# for a jackpot, or you can make the jackpot super by hitting all of the LSB targets 
# after finding Santa but before delivering him! This mode also overrides the normal
# functioning of the Hinterlands to give an alternative way of capturing Santa Claus.
# 
# Scenario
# Lock, Shock and Barrel are out to kidnap Santa Claus!
# 
# Details
# When this multiball starts, three major shots will be lit at random except for the Graveyard 
# and the left ramp, as these have special usage in this multiball. One of the major shots will 
# have Santa Claus, the other two will be bogus. Once you've got Santa Claus, the shot at the 
# graveyard lights and you must get past the gravestone and get to Jack to score the jackpot. 
# (You'll also get an add-a-ball if it's been qualified by the mystery target.) This process doesn't 
# change throughout the multiball, although there's a couple interesting points of note. Firstly, 
# it's possible to upgrade the jackpot into a super jackpot. To do this, you must hit all three of the LSB
# standups after kidnapping Santa Claus but before you deliver him to Jack. Secondly, the normal 
# functioning of the Hinterlands is overridden while this mode is active. Normally, starting a multiball 
# disables the Hinterlands Doors, but this mode keeps them going. If you shoot into the Hinterlands 
# with the wrong door selected, you get a small bonus and are treated to Jack saying you kidnapped the 
# wrong one. The slingshots will change the selected door as usual. When Christmas Town is selected, 
# the left ramp triangle will light up to indicate that shooting the Hinterlands will immediately 
# capture Santa Claus, guaranteed, but if you cycle the Hinterlands off of Christmas Town again, 
# the light will go out. Needless to say, this shot doesn't do anything while Santa Claus is captured. 
# As with all mutliballs, this mode is completed simply by starting it.
# 
# Scoring
# Wrong Major Shot                  100,000
# Wrong Hinterlands Door            100,000
# Santa Claus Kidnapped             250,000
# Jackpot                           5,000,000
# Jackpot Increase                  1,000,000
# Super Jackpot                     12,500,000
# Super Jackpot Increase            2,500,000
# 
# Lighting
# Major shots to aim for will flash with orange triangles. 
# The one with Santa Claus will actually be VERY slightly more yellow. 
# The Graveyard shot will flash a yellow triangle once ready for jackpot. 
# The LSB targets will flash red when hit before capturing Santa Claus and will begin 
# flashing white once Santa Claus is captured.
# Hitting them at this point makes them go solid white and once a super jackpot is ready, 
# the Graveyard shot will flash both the triangle and circle white. If the correct
# Hinterlands Door is selected to kidnap Santa Claus in this alternate manner, 
# the left ramp triangle will flash blue.
# 
# Difficulty Adjustments
# Very Easy        2 Major Shots Lit for Santa Claus
# Easy              2 Major Shots Lit for Santa Claus the First Time, 3 Each Subsequent Time
# Normal            3 Major Shots Lit for Santa Claus
# Hard              3 Major Shots Lit for Santa Claus the First Time, 4 Each Subsequent Time
# Very Hard         4 Major Shots Lit for Santa Claus
# 
# Char_LSB - LSB Multiball
# - completed LSB lights, enable locks
# - left ramp gate - opens bathtub diverter for 5 seconds
# - tub optos - count ball locks
# - 3 locks - start MB & drain sequence
# - diverter disabled when MB running
# - MB ends when complete or balls in play == 1

class Char_LSB(Mode):

    def mode_init(self):
        self.log.info( 'Char_LSB - LSB Multiball mode_init')

    def mode_start(self, **kwargs):
        self.log.info( 'LSB Multiball mode_start')
        self.delay = DelayManager()
        if self.player.lsb_multiball_mode_started == 0:
            #once per game only
            self.player.lsb_multiball_mode_started = 1
            self.player.lsb_physical_balls_locked = 0
            self.player.lsb_balls_locked = 0
        self.multiball_running = 0
        self.tub_lock_ready = 0
        self.santa_collected = 0
        self.add_mode_event_handler('ball_starting', self.count_balls_in_bathtub)
        self.add_mode_event_handler('sw_lrampgate', self.open_bathtub_diverter)
        self.add_mode_event_handler("lsb_standups_lsbp_lit_complete", self.lsb_complete)        
        self.add_mode_event_handler('sw_enter_tub', self.close_bathtub_diverter) #opto 3 triggered
        self.add_mode_event_handler('sw_enter_tub', self.settle_time)
        self.add_mode_event_handler('balldevice_trough_ball_enter', self.ball_drained)


    def settle_time(self, **kwargs):
        self.log.info("LSB - delay 1 second for settling")
        delaytime = Timing.string_to_ms('1s')
        self.delay.add(name='ball_entered', ms=delaytime, callback=self.ball_entered)


    def count_balls_in_bathtub(self, **kwargs):
        self.log.info('LSB - count balls in tub')
        self.player.lsb_physical_balls_locked = 0
        if self.machine.switch_controller.is_active('tublock1_opt'):
            self.player.lsb_physical_balls_locked += 1
        if self.machine.switch_controller.is_active('tublock2_opt'):
            self.player.lsb_physical_balls_locked += 1
        if self.machine.switch_controller.is_active('tublock3_opt'):
            self.player.lsb_physical_balls_locked += 1
        self.log.info('LSB - balls in tub = ' + str(self.player.lsb_physical_balls_locked))


    def open_bathtub_diverter(self, **kwargs):
        if (self.tub_lock_ready == 1 and self.multiball_running == 0):
            self.log.info("LSB - Bathtub Diverter open for 5 seconds")
            self.machine.coils['bathtubdiverter'].enable()
            delaytime = Timing.string_to_ms('5s')
            self.delay.add(name='close_bathtub_diverter', ms=delaytime, callback=self.close_bathtub_diverter)


    def close_bathtub_diverter(self, **kwargs):
        self.log.info( "LSB - Bathtub Diverter closed")
        self.machine.coils['bathtubdiverter'].disable()


    def ball_drained(self, **kwargs):
        if self.multiball_running == 1:
            self.log.info( "LSB - MB - ball drained" )
            self.log.info( "LSB - balls in play = " + str(self.machine.game.balls_in_play) )
            if self.machine.game.balls_in_play <= 2:
                if self.player.ball_save_active == 0:
                    #there is only 1 ball on playfield, end multiball
                    self.tub_lock_ready = 0
                    self.machine.events.post('tub_mb_stopping')
                    self.log.info( "LSB - MB over")
                    if self.santa_collected > 0:
                        self.machine.events.post('Char_mode_stopped', char_state="complete", char_mode="LSB")
                        self.multiball_running = 2 #completed
                    else:
                        self.machine.events.post('Char_mode_stopped', char_state="incomplete", char_mode="LSB")
                        self.multiball_running = 0 #ready to start again
                    self.machine.events.post('add_a_ball_stop')
                    self.machine.events.post('enable_combos')
                else:
                    self.log.info( "LSB - MB - ball drained - but ball save is running" )
            else:
                self.log.info( "LSB - MB - ball drained - more than 2 BIP" )            
                
                
    def lsb_complete(self, **kwargs):
        self.log.info( "LSB - lights completed")        	
        if self.multiball_running == 0:
            if self.tub_lock_ready == 0:
                self.log.info( "LSB - lights complete - enable tub lock")
                self.tub_lock_ready = 1
                self.flash_lock_ready()
                self.machine.events.post('LSB_bathtub_ready')
        else:
            self.log.info( "LSB - lights complete during MB - temp santa collected")        	
            self.santa_collected = 1


    def flash_lock_ready(self):
        #TODO - change speeds for lock 1, 2, 3
        self.log.info("LSB - Flash the bathtub lock light" )
        led = "rgb_lramp_arrow"
        script_name = "sc_green_flash"
        scp = self.machine.light_controller.registered_light_scripts[script_name]
        self.machine.light_controller.run_script(leds=led, script=scp, priority=50, tocks_per_sec=80, key="lock_lita", blend=True)


    def turn_off_lock_light(self):
        self.log.info('LSB - turn off lock light')
        self.machine.light_controller.stop_script("lock_lita")
        led = "rgb_lramp_arrow"
        script_name = "sc_off"
        scp = self.machine.light_controller.registered_light_scripts[script_name]
        self.machine.light_controller.run_script(leds=led, script=scp, priority=50, tocks_per_sec=80, key="lock_lita", repeat=0, blend=True)


    def ball_entered(self):
        self.log.info('LSB - ball entered tub')
        oldcount = self.player.lsb_physical_balls_locked
        self.count_balls_in_bathtub()
        if self.multiball_running == 0:
            self.player.lsb_balls_locked += 1
            self.machine.events.post('tub_ball_locked', lock=self.player.lsb_balls_locked)
            self.log.info('LSB - ball '+str(self.player.lsb_balls_locked)+' locked in tub')
            #more balls in tub, if not ours, drain instead of launching new one
            if self.player.lsb_physical_balls_locked > self.player.lsb_balls_locked:
                self.log.info('LSB - release excess ball')
                self.drain_tub()
            else:
                if self.player.lsb_balls_locked >= 3:
                    #this is the 3rd ball, start MB
                    self.machine.events.post('LSB_ball_3_locked')                    
                    self.startMB()
                else:
                    self.log.info('LSB - launch another ball')
                    delaytime = Timing.string_to_ms('2s')
                    self.delay.add(name='drain_bathtub', ms=delaytime, callback=self.add_another_ball)
        else:
            #we are in MB, why is there a ball in tub?  
            #the tub diverter should be closed too!
            self.drain_tub()

    def add_another_ball(self):
        self.machine.playfield.add_ball(balls=1, player_controlled=True)    
    
    def startMB(self):
        #start multiball
        self.log.info('LSB - start MB')
        self.multiball_running = 1
        self.santa_collected = 0
        self.player.lsb_balls_locked = 0
        if self.player.lsb_physical_balls_locked == 3:
            delaytime = Timing.string_to_ms('1s')
            self.delay.add(name='drain_bathtub', ms=delaytime, callback=self.slowrelease)
        if self.player.lsb_physical_balls_locked == 2:
            delaytime = Timing.string_to_ms('1s')
            self.delay.add(name='drain_bathtub', ms=delaytime, callback=self.slowrelease)
            self.log.info('LSB - 1 ball needed from trough')
            self.machine.playfield.add_ball(balls=1, player_controlled=False)
        if self.player.lsb_physical_balls_locked == 1:
            delaytime = Timing.string_to_ms('1s')
            self.delay.add(name='drain_bathtub', ms=delaytime, callback=self.slowrelease)
            self.log.info('LSB - 2 balls needed from trough')
            self.machine.playfield.add_ball(balls=2, player_controlled=False)
        self.machine.game.add_balls_in_play(2)
        self.machine.events.post('tub_mb_music_start')
        self.machine.events.post('add_a_ball_start') 
        self.machine.events.post('disable_combos')       
        self.turn_off_lock_light()


    def slowrelease(self):
        self.log.info('LSB - slow release')
        if self.player.lsb_physical_balls_locked > 0:
            self.drain_tub()
            delaytime = Timing.string_to_ms('2s')
            self.delay.add(name='drain_bathtub', ms=delaytime, callback=self.slowrelease)
        else: 
            self.log.info('LSB - tub is empty')        	


    def drain_tub(self):
        self.log.info('LSB - drain tub')
        self.machine.coils['bathtubdrain'].pulse(milliseconds=40)
        self.count_balls_in_bathtub()


    def mode_stop(self, **kwargs):
        self.log.info( 'LSB - mode_stop')
        self.delay.remove('close_bathtub_diverter')
        self.close_bathtub_diverter()
        self.turn_off_lock_light()
